Function Find-RDSConnectionBroker {

    # Get parameters
    Param(
        [parameter(Mandatory = $true, HelpMessage = "Please provide a valid object containing machines to search", position = 0)][object]$Machines,
        [parameter(Mandatory = $true, HelpMessage = "Please provide the name of the script calling this function", position = 1)][string]$ScriptName,
        [ValidateSet("Interactive", "Service")][parameter(Mandatory = $true, HelpMessage = "Please choose output mode", position = 2)][string]$OutputMode
    )

    BEGIN {

        # Setup variables.
        [bool]$FoundRDSConnectionBroker = $false
        [string]$OSVersion = (Get-OperatingSystem -ReturnType "MajorVersion")
        [string]$CBName = ""
    }

    PROCESS {

        # Loop though all computers in the domain to find RDS Session Broker Role.
        foreach ($Machine in $Machines) {
            if ((Test-Connection -ComputerName $Machine.Name -Quiet) -and ($FoundRDSConnectionBroker -eq $false)) {

                # log / display Event.
                Show-Output -Message "Hunting for RDS Connection Broker - $($Machine.Name)" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

                # Get installed roles (if supported).
                if ($OSVersion -ne "6.1") {

                    # Get list of installed features
                    [object]$InstalledRoles = Get-WindowsFeature -ComputerName $Machine.Name | Where-Object { $_.installstate -eq "installed" }

                    # Iterate through installed roles looking for RDS-Connection-Broker
                    foreach ($InstalledRole in $InstalledRoles) {
                        if ($InstalledRole.name -eq "RDS-Connection-Broker") {
                            $FoundRDSConnectionBroker = $true
                        }
                    }

                    #Check to see if we found the RDS ConnectionBroker, reset if we didn't find the role on this machine, if we did, the loop will exit.
                    if ($FoundRDSConnectionBroker -eq $true) {
                        $CBName = $Machine.Name
                    }
                    else {
                        # Reset objects / variables
                        Clear-Variable -Name "InstalledRoles"
                        $FoundRDSConnectionBroker = $false
                    }
                }
                else {
                    
                    # Return false as its a non-compatible system.
                    $FoundRDSConnectionBroker = $false
                }
            }
        }
    }

    END {

        if ($FoundRDSConnectionBroker -eq $true) {

            # log / display Event.
            Show-Output -Message "Found RDS Connection Broker: $($CBName)" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

            # Return Data.
            return $CBName
        }
        else {

            # log / display Event.
            Show-Output -Message "No RDS Connection Broker was found" -OutputFormat "Warning" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

            # Return Data.
            return "NOTFOUND"
        }
    }   
}

function Send-WarningMessage {

    # Get parameters
    Param(
        [parameter(Mandatory = $true, HelpMessage = "Please enter a valid countdown Duration (Minutes).", position = 0)][int]$Countdown,
        [parameter(Mandatory = $true, HelpMessage = "Please enter a valid connection broker.", position = 1)][string]$ConnectionBroker,
        [ValidateSet("Other", "Terminal", "Uptime")][parameter(Mandatory = $true, HelpMessage = "Please enter a valid operation mode.", position = 2)][string]$Operation_Mode,
        [parameter(Mandatory = $true, HelpMessage = "Please provide a valid object containing machines to search.", position = 3)][object]$Machines,
        [parameter(Mandatory = $true, HelpMessage = "Please enter a valid fully qualified domainname.", position = 4)][string]$FQDomainName,
        [parameter(Mandatory = $true, HelpMessage = "Please provide the name of the script calling this function.", position = 5)][string]$ScriptName,
        [ValidateSet("Interactive", "Service")][parameter(Mandatory = $true, HelpMessage = "Please choose output mode.", position = 6)][string]$OutputMode
    )

    BEGIN {

        # Get current time and date
        [datetime]$Time = Get-Date

        #Set delay time (closes warning before next one is due)
        if ($Countdown -gt 1) {
            [string]$Delaytime = "/time:$((($Countdown*60)/2) - 10)"
        }
        else {
            [string]$Delaytime = "/time:55"
        }

        # Build message to be sent to users.
        if ($Countdown -gt 1) {
            if ($Operation_Mode -eq "Terminal") {
                [string]$Message = "Please save your work and log off. This Server will be rebooted in $Countdown Minutes (Approximately: $($Time.AddMinutes($Countdown)))"
            }
            else {
                [string]$Message = "A server that you may rely on will be rebooted in $Countdown Minutes (Approximately: $($Time.AddMinutes($Countdown))), We recommend that you save your work and log off to avoid errors within applications you may have open."                
            }
        }
        else {
            if ($Operation_Mode -eq "Terminal") {
                [string]$Message = "Please save your work and log off. This Server will be rebooted in $Countdown Minute (60 Seconds). There will be NO further warnings."
            }
            else {
                [string]$Message = "A server that you may rely on will be rebooted in $Countdown Minute (60 Seconds), We recommend that you save your work and log off to avoid errors within applications you may have open. There will be NO further warnings."                
            }
        }
    }

    PROCESS {

        # Detect OS to see how we address warning the users. (2012+ can use RDS functionality, 2008r2 requires using old fahioned MSG)
        if (((Get-OperatingSystem -ReturnType "MajorVersion") -eq "6.1") -or ($ConnectionBroker -eq "NOTFOUND")) {
            
            # log / display Event
            Show-Output -Message "Sending Warning Message (2008 Mode) - $($Countdown) Minute warning" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

            # Loop through terminals, check they are online and send msg to all logged in users.
            foreach ($Machine in $Machines) {
                if (Test-Connection -ComputerName $Machine.Name -Quiet) {
                    Invoke-Command -ComputerName $Machine.Name -ArgumentList $Delaytime, $Message -scriptblock { msg * $args[0] $args[1] }
                }
            }    
        }
        else {
            
            # log / display Event
            Show-Output -Message "Sending Warning Message (2012+ Mode) - $($Countdown) Minute warning" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

            # Construct FQDN Name for Connection Broker and remove spaces.
            [string]$ConnectionBrokerFQDN = $ConnectionBroker + "." + $FQDomainName
            $ConnectionBrokerFQDN = $ConnectionBrokerFQDN.replace(" ", "")

            # Get all current user sessions.
            [object]$RDSessions = Get-RDUserSession -ConnectionBroker $ConnectionBrokerFQDN | Select-Object username, hostserver, unifiedsessionid
            
            # Iterate through all current RD User Sessions.
            foreach ($RDSession in $RDSessions) {

                # Send Message
                Send-RDUserMessage -HostServer $RDSession.hostserver -UnifiedSessionID $RDSession.unifiedsessionid -MessageTitle "Message from System Janitor to $($RDSession.username)" -MessageBody $Message
            }
        }
    }

    END { }
}

Function Switch-RDConnectionState {

    # Get parameters
    Param(
        [parameter(Mandatory = $true, HelpMessage = "Please enter a valid connection broker.", position = 0)][string]$ConnectionBroker,    
        [parameter(Mandatory = $true, HelpMessage = "Please provide a valid object containing machines to search", position = 1)][object]$Machines,
        [parameter(Mandatory = $true, HelpMessage = "Please enter a valid fully qualified domainname.", position = 2)][string]$FQDomainName,
        [ValidateSet("Permit", "Deny", "Turndown")][parameter(Mandatory = $true, HelpMessage = "Please enter a connection state.", position = 3)][string]$ConnectionState,
        [parameter(Mandatory = $true, HelpMessage = "Please provide the name of the script calling this function", position = 4)][string]$ScriptName,
        [ValidateSet("Interactive", "Service")][parameter(Mandatory = $true, HelpMessage = "Please choose output mode", position = 5)][string]$OutputMode
    )

    BEGIN {

        # Construct FQDN Name for Connection Broker and remove spaces.
        [string]$ConnectionBrokerFQDN = $ConnectionBroker + "." + $FQDomainName
        $ConnectionBrokerFQDN = $ConnectionBrokerFQDN.replace(" ", "")
    }

    PROCESS {

        # Iterate through machines that need to have connections set to disabled on chosen connection broker.
        foreach ($Machine in $Machines) {

            # Check to see if machine is alive
            if (Test-Connection -ComputerName $Machine.Name -Quiet) {

                # log / display Event
                Show-Output -Message "Processing new connection state for $($Machine.Name).$($FQDomainName)." -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0


                if ($ConnectionState -eq "Permit") {
                
                    # log / display Event
                    Show-Output -Message "Permitting connections to $($Machine.Name).$($FQDomainName)." -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

                    # Set terminal to not accept connections, but place back into pool at time of run to correct any previous actions
                    Set-RDSessionHost -SessionHost "$($Machine.Name).$($FQDomainName)" -NewConnectionAllowed Yes -ConnectionBroker "$($ConnectionBrokerFQDN)"
                } elseif ($ConnectionState -eq "Deny") {

                    # log / display Event
                    Show-Output -Message "Stopping connections to $($Machine.Name).$($FQDomainName)." -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

                    # Set terminal to not accept connections
                    Set-RDSessionHost -SessionHost "$($Machine.Name).$($FQDomainName)" -NewConnectionAllowed No -ConnectionBroker "$($ConnectionBrokerFQDN)"
                } elseif ($ConnectionState -eq "Turndown") {
                    
                    # log / display Event
                    Show-Output -Message "Stopping connections to $($Machine.Name).$($FQDomainName) until the next reboot (Turndown)." -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

                    # Set terminal to not accept connections, until the next reboot (script only option).
                    Set-RDSessionHost -SessionHost "$($Machine.Name).$($FQDomainName)" -NewConnectionAllowed NotUntilReboot -ConnectionBroker "$($ConnectionBrokerFQDN)"
                }
            }
            else {

                # log / display Event
                Show-Output -Message "$($Machine.Name).$($FQDomainName) offline, nothing can be done." -OutputFormat "Warning" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
            }
        }
    }

    END {}

    }