function Clear-Folder {

    # Get parameters
    Param(
        [parameter(Mandatory = $true, HelpMessage = "Please provide a machine for this task to be carried out on (Machine Name / LocalHost)", position = 0)][string]$Machine,
        [ValidateSet("Logs", "PerfLogs", "Temp", "Usr")][parameter(Mandatory = $true, HelpMessage = "Please provide a folder name to be cleared / processed (Logs / PerfLogs / Usr / Temp)", position = 1)][string]$FolderName,
        [parameter(Mandatory = $true, HelpMessage = "Please provide the name of the script calling this function", position = 2)][string]$ScriptName,
        [ValidateSet("Interactive", "Service")][parameter(Mandatory = $true, HelpMessage = "Please choose output mode", position = 3)][string]$OutputMode
    )

    BEGIN {

        # log / display Event.
        Show-Output -Message "Clearing folder: $($FolderName) on machine: $($Machine)" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

        # Set folder path to be processed.
        if ($FolderName -eq "Logs") {
            
            # Store folder path
            $FolderPath = "C:\Logs"
        }
        elseif ($FolderName -eq "PerfLogs") {
        
            # Store folder path
            $FolderPath = "C:\PerfLogs"
        }
        elseif ($FolderName -eq "Temp") {
        
            # Store folder path
            $FolderPath = "C:\Temp"
        }
        elseif ($FolderName -eq "Usr") {
        
            # Store folder path
            $FolderPath = "C:\Usr"
        }
    }

    PROCESS {

        # Loop through all terminal servers in the domain, check they are online and clean folders.
        if ($Machine -ne "LocalHost") {

            # log / display Event.
            Show-Output -Message "Attempting to clear folder: $($FolderName) on $($Machine)" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

            if (Test-Connection -ComputerName $Machine -Quiet) {
                
                # log / display Event.
                Show-Output -Message "$($Machine) Found, attempting to clear folder: $($FolderName)" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

                # Setup session (will run on remote machine)
                [object]$RSession = New-PSSession -ComputerName $Machine

                # Invoke command on remote session (All code in script block executes on remote machine), store return vale in variable.
                Invoke-Command -Session $RSession -ScriptBlock {

                    # [REMOTE] Remove named folder from disk
                    Remove-Item $using:FolderPath -Recurse -Force -Confirm:$false -ErrorAction SilentlyContinue | out-null
                }

                # log / display Event.
                Show-Output -Message "Cleared folder $($FolderName) from $($Machine)" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
            }
            else {

                # log / display Event.
                Show-Output -Message "$($Machine) not found." -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
            }
        }
        else {

            # Remove named folder from disk
            Remove-Item $($FolderPath) -Recurse -Force -Confirm:$false -ErrorAction SilentlyContinue | out-null

            # log / display Event.
            Show-Output -Message "Cleared folder $($FolderName) from $($Machine)" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
        }
    }

    END {

        # Remove sessions in use as they are no longer needed.
        Get-PSSession | Remove-PSSession
    }
}

function Clear-Profiles {

    # Get parameters
    Param(
        [parameter(Mandatory = $true, HelpMessage = "Please provide a valid object containing machines to search", position = 0)][object]$Machines,
        [parameter(Mandatory = $true, HelpMessage = "Please provide the name of the script calling this function", position = 1)][string]$ScriptName,
        [ValidateSet("Interactive", "Service")][parameter(Mandatory = $true, HelpMessage = "Please choose output mode", position = 2)][string]$OutputMode
    )
    
    BEGIN {

        # Setup variables.
        [string]$DelProfApp = "C:\Maintenance\delprof2.exe"
        [string]$Unattended = "/u"
        [string]$RemoteOnly = "/r"

        # Get registry control settings (THese are pushed out via group policy on the platform (OP Sec Policy))
        [string]$BristleType = Get-RegistryValue -RegKey "HKLM:\Software\Accesspoint\Janitor" -RegValue "JanitorBrushBristles" -FallbackValue "Soft"
    }

    PROCESS {

        # Loop through terminals, check they are online and clean stale profiles.
        foreach ($Machine in $Machines) {
            if (Test-Connection -ComputerName $Machine.Name -Quiet) {
                
                #Setup computer variable.
                [string]$Computer = "/c:$($Machine.Name)"

                #Execute command locally (relies on having delprof2 in the same folder).
                if ($BristleType -eq "Soft") {

                    # log / display Event.
                    Show-Output -Message "Clearing Profiles - Using Soft Bristled Brush on $($Machine.Name)" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

                    # Run delprof
                    Start-Process -FilePath $DelProfApp -ArgumentList $Computer, $Unattended, $RemoteOnly
                }
                else {

                    # log / display Event.
                    Show-Output -Message "Clearing Profiles - Using Hard Bristled Brush on $($Machine.Name)" -OutputFormat "Warning" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
                    
                    # Run delprof
                    Start-Process -FilePath $DelProfApp -ArgumentList $Computer, $Unattended
                }
            }
        }
    }

    END { }
}

function Clear-RegistryBloat {

    # Get parameters
    Param(
        [parameter(Mandatory = $true, HelpMessage = "Please provide a valid object containing machines to search", position = 0)][object]$Machines,
        [parameter(Mandatory = $true, HelpMessage = "Please provide a date of the month to run on", position = 1)][int]$RunDay,
        [parameter(Mandatory = $true, HelpMessage = "Please provide the actual day of the month", position = 2)][int]$ActualDay,
        [parameter(Mandatory = $true, HelpMessage = "Please provide the name of the script calling this function", position = 3)][string]$ScriptName,
        [ValidateSet("Interactive", "Service")][parameter(Mandatory = $true, HelpMessage = "Please choose output mode", position = 4)][string]$OutputMode
    )

    BEGIN {

        # Check to see if function is to run.
        if ($ActualDay -eq 1) {
            [bool]$RunRegistryClean = $true
        }
        else {
            [bool]$RunRegistryClean = $false
        }

        # Check to see if the registry needs cleaning.
        if ($RunRegistryClean -eq $true) {

            # log / display Event.
            Show-Output -Message "Clearing Firewall Registry Bloat. Operation Mode: $($Operation_Mode)" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
        }
        else {

            # log / display Event.
            Show-Output -Message "Skipping Firewall Registry Bloat cleanup. Operation Mode: $($Operation_Mode)" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
        }
    }

    PROCESS {

        # Loop through terminals, check they are online and clean registry bloat (if needed).
        if ($RunRegistryClean -eq $true) {
            foreach ($Machine in $Machines) {
                if (Test-Connection -ComputerName $Machine.Name -Quiet) {
                
                    # log / display Event.
                    Show-Output -Message "Attempting to clear firewall registry bloat from $($Machine.Name)" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

                    # Setup session (will run on remote machine)
                    [object]$RSession = New-PSSession -ComputerName $Machine.Name

                    # Invoke command on remote session (All code in script block executes on remote machine).
                    Invoke-Command -Session $RSession -ScriptBlock {
                    
                        # [REMOTE] Delete known oversized reg key that causes slow login for the first user after maintenance.
                        reg delete HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Services\SharedAccess\Parameters\FirewallPolicy\FirewallRules /va /f
                        reg delete HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Services\SharedAccess\Parameters\FirewallPolicy\RestrictedServices\Configurable\System /va /f
                        reg delete HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Services\SharedAccess\Parameters\FirewallPolicy\RestrictedServices\AppIso\FirewallRules /va /f
                    } -AsJob -JobName "ClearRegistry_$($Machine.Name)"
                }
            }

            # Pause until all done
            Get-Job | Wait-Job

            # Get job details.
            [object]$DoneJobs = Get-Job

            # Loop through all jobs and extract returned results from machines
            foreach ($DoneJob in $DoneJobs) {

                # Recreate machine name
                $Machine = $DoneJob.Name
                $Machine = $Machine.replace("ClearRegistry_", "")

                # log / display Event.
                Show-Output -Message "Cleared firewall registry bloat from $($Machine) as part of job $($DoneJob.Name)" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
            }

            # Clear out all old jobs.
            Get-Job | Remove-Job

            # Remove sessions in use as they are no longer needed.
            Get-PSSession | Remove-PSSession
        }
    }

    END { }    
}

function Clear-TeamsSetupFolders {
    
    # Get parameters
    Param(
        [parameter(Mandatory = $true, HelpMessage = "Please provide a valid object containing machines to search", position = 0)][object]$Machines,
        [parameter(Mandatory = $true, HelpMessage = "Please provide the name of the script calling this function", position = 1)][string]$ScriptName,
        [ValidateSet("Interactive", "Service")][parameter(Mandatory = $true, HelpMessage = "Please choose output mode", position = 2)][string]$OutputMode
    )

    BEGIN {

        # log / display Event.
        Show-Output -Message "Clearing Teams Setup Folders" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
    }

    PROCESS {

        # Loop through all terminal servers in the domain, check they are online and clean folders.
        foreach ($Machine in $Machines) {
            if (Test-Connection -ComputerName $Machine.Name -Quiet) {
                
                # log / display Event.
                Show-Output -Message "Attempting to clear Teams Setup Folders from $($Machine.Name)" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

                # Setup session (will run on remote machine)
                [object]$RSession = New-PSSession -ComputerName $Machine.Name

                # Invoke command on remote session (All code in script block executes on remote machine), store return vale in variable.
                $RemoteResult = Invoke-Command -Session $RSession -ScriptBlock {

                    # [REMOTE] Setup variables
                    [string]$WindowsTeamsSetupFolder = "C:\Users\All Users"

                    # [REMOTE] Enumerate files within folder that match selection criteria.
                    [object]$Folders = Get-ChildItem $WindowsTeamsSetupFolder -Recurse -Filter *squirrel* -Directory | Select-Object FullName


                    # [REMOTE] Iterate through files.
                    foreach ($Folder in $Folders) {
                        Remove-Item $Folder.FullName -Recurse -ErrorAction SilentlyContinue -Confirm:$false -Force | out-null
                    }

                    # [REMOTE] Return file count
                    $Folders.count | Out-String
                } -AsJob -JobName "ClearTeamsSetup_$($Machine.Name)"
            }
        }

        # Pause until all done
        Get-Job | Wait-Job

        # Get job details.
        [object]$DoneJobs = Get-Job

        # Loop through all jobs and extract returned results from machines
        foreach ($DoneJob in $DoneJobs) {

            # Extract results
            $RemoteResult = Receive-Job -Name $DoneJob.Name
    
            # Correct results if required.
            if ($RemoteResult -eq "") {
                $RemoteResult = "0"
            }

            # Recreate machine name
            $Machine = $DoneJob.Name
            $Machine = $Machine.replace("ClearTeamsSetup_", "")

            # log / display Event.
            Show-Output -Message "Cleared approximately $($RemoteResult) Teams Setup Folders from $($Machine) as part of job $($DoneJob.Name)" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
        }

        # Clear out all old jobs.
        Get-Job | Remove-Job

        # Remove sessions in use as they are no longer needed.
        Get-PSSession | Remove-PSSession
    }

    END { }
}

function Clear-TempFiles {

    # Get parameters
    Param(
        [parameter(Mandatory = $true, HelpMessage = "Please provide a valid object containing machines to search", position = 0)][object]$Machines,
        [parameter(Mandatory = $true, HelpMessage = "Please provide the name of the script calling this function", position = 1)][string]$ScriptName,
        [ValidateSet("Interactive", "Service")][parameter(Mandatory = $true, HelpMessage = "Please choose output mode", position = 2)][string]$OutputMode
    )

    BEGIN {

        # log / display Event.
        Show-Output -Message "Clearing Temp Files" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
    }

    PROCESS {

        # Loop through all machines in the domain, check they are online and clean folders.
        foreach ($Machine in $Machines) {
            if (Test-Connection -ComputerName $Machine.Name -Quiet) {
                
                # log / display Event.
                Show-Output -Message "Attempting to clear Temp Files from $($Machine.Name)" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

                # Setup session (will run on remote machine)
                [object]$RSession = New-PSSession -ComputerName $Machine.Name

                # Invoke command on remote session (All code in script block executes on remote machine), store return vale in variable.
                $RemoteResult = Invoke-Command -Session $RSession -ScriptBlock {

                    # [REMOTE] Setup variables
                    [string]$WindowsTempFolder = "C:\Windows\Temp"

                    # [REMOTE] Enumerate files within folder that match selection criteria.
                    [object]$Files = Get-ChildItem $WindowsTempFolder -Recurse | Select-Object FullName

                    # [REMOTE] Iterate through files.
                    foreach ($File in $Files) {
                        Remove-Item $File.FullName -Recurse -ErrorAction SilentlyContinue -Confirm:$false -Force | out-null
                    }

                    # [REMOTE] Return file count
                    $Files.count | Out-String
                } -AsJob -JobName "ClearTemp_$($Machine.Name)"
            }
        }

        # Pause until all done
        Get-Job | Wait-Job

        # Get job details.
        [object]$DoneJobs = Get-Job

        # Loop through all jobs and extract returned results from machines
        foreach ($DoneJob in $DoneJobs) {

            # Extract results
            $RemoteResult = Receive-Job -Name $DoneJob.Name
    
            # Correct results if required.
            if ($RemoteResult -eq "") {
                $RemoteResult = "0"
            }

            # Recreate machine name
            $Machine = $DoneJob.Name
            $Machine = $Machine.replace("ClearTemp_", "")

            # log / display Event.
            Show-Output -Message "Cleared approximately $($RemoteResult) Temp Files from $($Machine) as part of job $($DoneJob.Name)" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
        }

        # Clear out all old jobs.
        Get-Job | Remove-Job

        # Remove sessions in use as they are no longer needed.
        Get-PSSession | Remove-PSSession
    }

    END { }
}

function Restart-Machines {

    # Get parameters
    Param(
        [parameter(Mandatory = $true, HelpMessage = "Please provide a valid object containing machines to search", position = 0)][object]$Machines,
        [parameter(Mandatory = $true, HelpMessage = "Please provide the name of the script calling this function", position = 1)][string]$ScriptName,
        [ValidateSet("Interactive", "Service")][parameter(Mandatory = $true, HelpMessage = "Please choose output mode", position = 2)][string]$OutputMode
    )

    BEGIN {

        # log / display Event.
        Show-Output -Message "Restarting Machines" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
    }

    PROCESS {

        # Loop through terminals, check they are online and reboot them.
        foreach ($Machine in $Machines) {
            if (Test-Connection -ComputerName $Machine.Name -Quiet) {

                # log / display Event.
                Show-Output -Message "Restarting Machines - $($Machine.Name)" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

                # Send restart job
                Restart-Computer -ComputerName $Machine.Name -Force -AsJob
            }
        }
    }

    END { }
}

function Set-UserAttributes {

    # Get parameters
    Param(
        [Parameter(Mandatory = $true, HelpMessage = "Object containing users to process", Position = 0)][object]$Users,
        [ValidateSet("City", "Company", "Country", "Department", "Description", "Fax", "HomePage", "HomePhone", "MobilePhone", "Office", "OfficePhone", "Organization", "Pager", "POBox", "PostalCode", "State", "StreetAddress")][parameter(Mandatory = $true, HelpMessage = "Please enter a source attribute", position = 1)][string]$SourceAttribute,
        [ValidateSet("City", "Company", "Country", "Department", "Description", "Fax", "HomePage", "HomePhone", "MobilePhone", "Office", "OfficePhone", "Organization", "Pager", "POBox", "PostalCode", "State", "StreetAddress")][parameter(Mandatory = $true, HelpMessage = "Please enter a destination attribute", position = 2)][string]$DestinationAttribute,
        [ValidateSet("Copy", "Move", "Swap")][parameter(Mandatory = $true, HelpMessage = "Please enter a format to display output", position = 3)][string]$OperationMode,
        [parameter(Mandatory = $true, HelpMessage = "Please provide the name of the script calling this function", position = 4)][string]$ScriptName,
        [ValidateSet("Interactive", "Service")][parameter(Mandatory = $true, HelpMessage = "Please choose output mode", position = 5)][string]$OutputMode
    )

    BEGIN {

        # Setup variables.
        [object]$TranslateMatrix = @()
        [bool]$AttributesMatch = $false
        [object]$SourceAttributeData = ""
        [object]$SourceAttributeProperty | Out-Null
        [bool]$SourceAttributeBlank = $false
        [bool]$DestinationAttributeBlank = $false
        [object]$DestinationAttributeData = ""
        [object]$DestinationAttributeProperty | Out-Null

        # Check to see if user has entered the same attribute name for source and destination.
        if ($($SourceAttribute) -eq $($DestinationAttribute)) {
            
            # Mark as matched.
            $AttributesMatch = $true
        }

        # Build translation matrix.
        $TranslateMatrix += [PSCustomObject]@{Regular = "City"; LDAP = "City" }
        $TranslateMatrix += [PSCustomObject]@{Regular = "Company"; LDAP = "Company" }
        $TranslateMatrix += [PSCustomObject]@{Regular = "Country"; LDAP = "co" }
        $TranslateMatrix += [PSCustomObject]@{Regular = "Department"; LDAP = "Department" }
        $TranslateMatrix += [PSCustomObject]@{Regular = "Description"; LDAP = "Description" }
        $TranslateMatrix += [PSCustomObject]@{Regular = "Fax"; LDAP = "facsimiletelephonenumber" }
        $TranslateMatrix += [PSCustomObject]@{Regular = "HomePage"; LDAP = "wwwhomepage" }
        $TranslateMatrix += [PSCustomObject]@{Regular = "HomePhone"; LDAP = "HomePhone" }
        $TranslateMatrix += [PSCustomObject]@{Regular = "MobilePhone"; LDAP = "mobile" }
        $TranslateMatrix += [PSCustomObject]@{Regular = "Office"; LDAP = "physicaldeliveryofficename" }
        $TranslateMatrix += [PSCustomObject]@{Regular = "OfficePhone"; LDAP = "OfficePhone" }
        $TranslateMatrix += [PSCustomObject]@{Regular = "Organization"; LDAP = "Organization" }
        $TranslateMatrix += [PSCustomObject]@{Regular = "Pager"; LDAP = "pager" }
        $TranslateMatrix += [PSCustomObject]@{Regular = "POBox"; LDAP = "postofficebox" }
        $TranslateMatrix += [PSCustomObject]@{Regular = "PostalCode"; LDAP = "PostalCode" }
        $TranslateMatrix += [PSCustomObject]@{Regular = "State"; LDAP = "st" }
        $TranslateMatrix += [PSCustomObject]@{Regular = "StreetAddress"; LDAP = "StreetAddress" }
    }

    PROCESS {

        # Process if source and destination attributes are different.
        if ($AttributesMatch -eq $false) {

            # Loop through users.
            foreach ($User in $Users) {

                # log / display Event.
                Show-Output -Message "Processing user: $($User.Name) ($($User.SamAccountName))" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

                # Get source attribute data.
                $SourceAttributeData = Get-ADUser -Identity $User.SamAccountName -Properties $SourceAttribute

                # Check it has data, flag if not.
                if ($SourceAttributeData.$SourceAttribute.length -eq 0) {

                    # Mark as blank
                    $SourceAttributeBlank = $true
                }

                # Check to see if source data is available (Limits operation modes)
                if ($SourceAttributeBlank -eq $true) {
                    
                    # log / display Event.
                    Show-Output -Message "The selected attribute ($($SourceAttribute)) for user: $($User.Name) ($($User.SamAccountName)) is blank, NO further processing will be carried out." -OutputFormat "Error" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
                }
                else {

                    # Process based on modes chosen
                    if ($OperationMode -eq "Copy") {

                        # Copy data to new variable.
                        $DestinationAttributeData = $SourceAttributeData

                        # Setup destination items.
                        $DestinationAttributeProperty = $TranslateMatrix | Where-Object { $_.Regular -eq "$($DestinationAttribute)" } | Select-Object "LDAP"

                        # Store data.
                        Set-ADUser -Identity $User.SamAccountName -Replace @{$DestinationAttributeProperty.ldap = $DestinationAttributeData.$($SourceAttribute) }
                        
                        # log / display Event.
                        Show-Output -Message "The selected attribute ($($SourceAttribute)) data for user: $($User.Name) ($($User.SamAccountName)) has been copied to the destination attribute ($($DestinationAttribute))" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
                    }
                    elseif ($OperationMode -eq "Move") {

                        # Copy data to new variable.
                        $DestinationAttributeData = $SourceAttributeData

                        # Setup source items.
                        $SourceAttributeProperty = $TranslateMatrix | Where-Object { $_.Regular -eq "$($SourceAttribute)" } | Select-Object "LDAP"

                        # Setup destination items.
                        $DestinationAttributeProperty = $TranslateMatrix | Where-Object { $_.Regular -eq "$($DestinationAttribute)" } | Select-Object "LDAP"

                        # Store data.
                        Set-ADUser -Identity $User.SamAccountName -Replace @{$DestinationAttributeProperty.ldap = $DestinationAttributeData.$($SourceAttribute) } -Clear $SourceAttributeProperty.ldap
                        
                        # log / display Event.
                        Show-Output -Message "The selected attribute ($($SourceAttribute)) data for user: $($User.Name) ($($User.SamAccountName)) has been moved to the destination attribute ($($DestinationAttribute))" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
                    }
                    elseif ($OperationMode -eq "Swap") {

                        # Get destination attribute data.
                        $DestinationAttributeData = Get-ADUser -Identity $User.SamAccountName -Properties $DestinationAttribute

                        # For swap, check both values have data.
                        # Check destination has data, flag if not.
                        if ($DestinationAttributeData.$DestinationAttribute.length -eq 0) {

                            # Mark as blank
                            $DestinationAttributeBlank = $true
                        }
                        else {
                            
                            $DestinationAttributeBlank = $false
                        }

                        # log / display Event.
                        Show-Output -Message "Attribute modification in progress (Source: $($SourceAttribute), Destination: $($DestinationAttribute))" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

                        # Setup source items.
                        $SourceAttributeProperty = $TranslateMatrix | Where-Object { $_.Regular -eq "$($SourceAttribute)" } | Select-Object "LDAP"

                        # Setup destination items.
                        $DestinationAttributeProperty = $TranslateMatrix | Where-Object { $_.Regular -eq "$($DestinationAttribute)" } | Select-Object "LDAP"

                        # Process if both sides have data.
                        if (($SourceAttributeBlank -eq $false) -and ($DestinationAttributeBlank -eq $false)) {

                            # Store data.
                            Set-ADUser -Identity $User.SamAccountName -Replace @{$DestinationAttributeProperty.ldap = $SourceAttributeData.$($SourceAttribute); $SourceAttributeProperty.ldap = $DestinationAttributeData.$($DestinationAttribute) }

                            # log / display Event.
                            Show-Output -Message "The selected attribute ($($SourceAttribute)) data for user: $($User.Name) ($($User.SamAccountName)) has been swapped with the destination attribute ($($DestinationAttribute)) data" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
                        }
                        else {

                            # log / display Event.
                            Show-Output -Message "The selected attribute ($($SourceAttribute)) data for user: $($User.Name) ($($User.SamAccountName)) has NOT been swapped with the destination attribute ($($DestinationAttribute)) data" -OutputFormat "Error" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
                        }
                    }
                }
            }
        }
        else {

            # log / display Event.
            Show-Output -Message "The selected source attribute ($($SourceAttribute)) and destination attribute ($($DestinationAttribute)) match, NO processing was carried out" -OutputFormat "Error" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
        }
    }

    END {}
}

function Start-ChocoMultiMode {
    
    # Get parameters
    Param(
        [parameter(Mandatory = $true, HelpMessage = "Please provide a valid object containing machines to run command on", position = 0)][object]$Machines,
        [parameter(Mandatory = $true, HelpMessage = "Please provide the name of the script calling this function", position = 1)][string]$ScriptName,
        [ValidateSet("Install", "Upgrade", "Uninstall")][parameter(Mandatory = $true, HelpMessage = "Mode to run choco in", position = 2)][string]$ChocoMode,
        [ValidateSet("Forced", "Standard")][parameter(Mandatory = $true, HelpMessage = "Does choco need to run as forced", position = 3)][string]$ChocoForce,
        [parameter(Mandatory = $true, HelpMessage = "Please provide the name of the choco package to execute", position = 4)][string]$ChocoPackage,
        [parameter(Mandatory = $false, HelpMessage = "Please provide the choco arguments to pass (if required)", position = 5)][string]$ChocoArguments,
        [ValidateSet("Interactive", "Service")][parameter(Mandatory = $true, HelpMessage = "Please choose output mode", position = 6)][string]$OutputMode
    )

    BEGIN {

        # Setup variables.
        [string]$ChocoPackageCheck = ""
        [bool]$ChocoPackageValid = $false

        # log / display Event.
        Show-Output -Message "Installing Choco package: $($ChocoPackage)" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

        # Check that choco package exists.
        $ChocoPackageCheck = choco search $($ChocoPackage) -r

        # Check output to see if package is valid.
        if ($ChocoPackageCheck.length -gt 0) {
            $ChocoPackageValid = $true
        }
    }

    PROCESS {

        # Check to see if package is valid and the installation is to progress
        if ($($ChocoPackageValid) -eq $true) {

            # log / display Event.
            Show-Output -Message "Choco package: $($ChocoPackage) is valid, proceeding with installation" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

            # Loop through all specified machines in the domain, check they are online and run choco command.
            foreach ($Machine in $Machines) {

                # Check connection to machine that is to be processed, only run if machine is online.
                if (Test-Connection -ComputerName $Machine.Name -Quiet) {
                    
                    if ($ChocoForce -eq "Forced") {

                        # log / display Event.
                        Show-Output -Message "Attempting to $($ChocoMode) choco package $($ChocoPackage) [Forced] on $($Machine.Name)" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

                        # Setup session (will run on remote machine)
                        [object]$RSession = New-PSSession -ComputerName $Machine.Name

                        # Invoke command on remote session (All code in script block executes on remote machine).
                        Invoke-Command -Session $RSession -ScriptBlock {

                            # [REMOTE] run choco upgrade command (Forced)
                            choco $Using:ChocoMode $using:ChocoPackage -y $using:ChocoArguments --force
                            
                        } -AsJob -JobName "ChocoMultiMode_$($Machine.Name)"
                    }
                    else {

                        # log / display Event.
                        Show-Output -Message "Attempting to $($ChocoMode) choco package $($ChocoPackage) [Standard] on $($Machine.Name)" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

                        # Setup session (will run on remote machine)
                        [object]$RSession = New-PSSession -ComputerName $Machine.Name

                        # Invoke command on remote session (All code in script block executes on remote machine).
                        Invoke-Command -Session $RSession -ScriptBlock {

                            # [REMOTE] run choco upgrade command (Standard)
                            choco $Using:ChocoMode $using:ChocoPackage -y $using:ChocoArguments
                            
                        } -AsJob -JobName "ChocoMultiMode_$($Machine.Name)"
                    }
                }
                else {

                    # log / display Event.
                    Show-Output -Message "Machine: $($Machine.Name) not available at this time." -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
                }
            }

            # Pause until all done
            Get-Job | Wait-Job

            # Get job details.
            [object]$DoneJobs = Get-Job

            # Loop through all jobs and extract returned results from machines
            foreach ($DoneJob in $DoneJobs) {

                # Recreate machine name
                $Machine = $DoneJob.Name
                $Machine = $Machine.replace("ChocoMultiMode_", "")

                # log / display Event.
                Show-Output -Message "Installed package $($ChocoPackage) on machine $($Machine) as part of job $($DoneJob.Name)" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
            }

            # Clear out all old jobs.
            Get-Job | Remove-Job

            # Remove sessions in use as they are no longer needed.
            Get-PSSession | Remove-PSSession

        }
        else {
            
            # log / display Event.
            Show-Output -Message "Choco package: $($ChocoPackage) is NOT valid, installation will NOT proceed." -OutputFormat "Warning" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
        }
    }

    END { }
}

function Stop-Machines {

    # Get parameters
    Param(
        [parameter(Mandatory = $true, HelpMessage = "Please provide a valid object containing machines to search / process", position = 0)][object]$Machines,
        [parameter(Mandatory = $true, HelpMessage = "Please provide a valid operation mode", position = 1)][object]$Operation_Mode,
        [parameter(Mandatory = $true, HelpMessage = "Please provide the name of the script calling this function", position = 2)][string]$ScriptName,
        [ValidateSet("Interactive", "Service")][parameter(Mandatory = $true, HelpMessage = "Please choose output mode", position = 3)][string]$OutputMode
    )

    BEGIN { 

        # log / display Event.
        Show-Output -Message "Stopping Machines" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
    }

    PROCESS {

        # Loop through machines, check they are online and reboot them.
        foreach ($Machine in $Machines) {

            # log / display Event.
            Show-Output -Message "Processing machine: $($Machine.Name)." -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

            if (Test-Connection -ComputerName $Machine.Name -Quiet) {
                if ($Operation_Mode -eq "Soft") {
                    
                    # log / display Event.
                    Show-Output -Message "Attempting to put machine $($Machine.Name) to sleep." -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
                    
                    # Attempt to stop computer.
                    Stop-Computer -ComputerName $Machine.Name
                }
                else {

                    # log / display Event.
                    Show-Output -Message "Putting machine $($Machine.Name) to sleep. (Forced)" -OutputFormat "Warning" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
                    
                    # Force stop computer.
                    Stop-Computer -ComputerName $Machine.Name -Force
                }
            }
            else {

                # log / display Event.
                Show-Output -Message "Machine $($Machine.Name) already sleeping." -OutputFormat "Warning" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
            }
        }
    }

    END { }
}

function Update-APAdminUsers {

    # Get parameters
    Param(
        [parameter(Mandatory = $true, HelpMessage = "Please provide the name of the script calling this function", position = 0)][string]$ScriptName,
        [ValidateSet("Interactive", "Service")][parameter(Mandatory = $true, HelpMessage = "Please choose output mode", position = 1)][string]$OutputMode,
        [ValidateSet("Cleanup", "LeaveBehind")][parameter(Mandatory = $true, HelpMessage = "Please choose Exit mode", position = 2)][string]$ExitMode
    )

    BEGIN {

        # Setup variables.
        [string]$ComputerName = $env:COMPUTERNAME
        [bool]$OnDomainController = $false

        # Check we are on a permitted machine for this function. (1st DC in domain)
        if (Test-MachineName "Filter" "DomainController" $ComputerName) {
            
            # Mark that we are on correct machine.
            $OnDomainController = $true

            # Get domain specific variables.
            [string]$DomainDName = Get-ADDomain | Select-Object -ExpandProperty DistinguishedName
            [string]$DomainFName = Get-ADDomain | Select-Object -ExpandProperty forest
        }

        # Setup additional variables.
        [string]$RoadieGUID = "fdbe5abe-7f38-4fd9-88bd-008d28baafb6"
        [string]$RoadieResourceBundleUrl = "https://deploy.theaccesspoint.co.uk/$($RoadieGUID)/resources.zip"
        [string]$RoadieTemp = "C:\$($RoadieGUID)\roadie.zip"
        [string]$RoadieResourceBundleOutput = "C:\$($RoadieGUID)"
        [string]$APAdminList = "C:\$RoadieGUID\apadmin.csv"
        [string]$APRemoveList = "C:\$RoadieGUID\apremove.csv"
        [string]$AdminSharedFolder = "C:\Adminshared"
        [bool]$APAdminPresent = $false
        [bool]$APremovePresent = $false
        
        # Process this block if we are on a domain controller.
        if ($OnDomainController -eq $true) {

            # Client email domain from registry
            [string]$ClientEmailDomain = Get-RegistryValue -RegKey "HKLM:\SOFTWARE\Accesspoint\Roadie" -RegValue "ClientEmailDomain" -FallbackValue "N/A"

            # Get details of client email domain name if nothing is returned from the registry.
            if ($ClientEmailDomain -eq "N/A") {
             
                $ClientEmailDomain = Confirm-InputData "General" "Please %toggle% the Clients email domain (eg lawclient.co.uk)" "The Names must match to continue."
            }
		
            # Store client email in the registry (admin user routines reference this)
            Edit-RegistryValue -EditMode "Amend" -RegKey "HKLM:\SOFTWARE\Accesspoint\Roadie" -RegValue "ClientEmailDomain" -RegValueData "$ClientEmailDomain" -RegValueType "String"
            
            # Check for required files in guid folder (Roadie) - flag status
            $APAdminPresent = Test-Path -Path "$($APAdminList)" -PathType Leaf
            $APRemovePresent = Test-Path -Path "$($APRemoveList)" -PathType Leaf

            # Download and extract if file not present.
            if (($APAdminPresent -eq $false) -or ($APremovePresent -eq $false)) {

                # Setup scratch folders.
                Write-Host "Setting up Scratch Folders" -ForegroundColor Green
                New-Item -ItemType Directory -Force -Path "$($RoadieTemp)"

                # log / display Event.
                Show-Output -Message "Files are NOT present in expected location, downloading and processing resource file." -OutputFormat "Warning" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

                # Build Resource Splat hash.
                $ResourceSplathash = @{
                    ScriptName        = "$($ScriptName)"
                    OutputMode        = $OutputMode
                    DownloadURL       = "$($RoadieResourceBundleUrl)"
                    DownloadLocation  = "$($RoadieTemp)"
                    ExtractionPath    = "$($RoadieResourceBundleOutput)"
                    ResourcePackName  = "Roadie"
                }

                # Download and extract resource pack
                Initialize-ResourcePack @ResourceSplathash
            }
            else {

                # log / display Event.
                Show-Output -Message "Files are present in expected location, no need to download resource file." -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
            }
        }
    }

    PROCESS {

        # Process this block if we are on a domain controller.
        if ($OnDomainController -eq $true) {

            # Add line break, tell the users wnhats going on
            Write-Host ""
            Write-Host "Adding new users, correcting existing users should they need it" -ForegroundColor Green
            Write-Host ""

            # Process files.
            # Import list into object for further processing
            [object]$APAdminUsers = Import-Csv -Path $APAdminList -Delimiter ","

            # Iterate through the list and Add users that are not present, and cleanup names on existing users
            foreach ($User in $APAdminUsers) {

                # Check to see if user exists on system
                if (Test-UserExists -Username "$($User.Username)" -SearchMode "Local") {

                    # Build Splat hash.
                    $Splathash = @{
                        GivenName     = $User.fname
                        Surname       = $User.lname
                        Displayname   = "$($User.fname) $($User.lname)"
                        EmailAddress  = "$($user.username)@$($ClientEmailDomain)"
                        Homedrive     = "H:"
                        ProfilePath   = "\\$($ComputerName)\adminprofiles$\$($User.username)"
                        HomeDirectory = "\\$($ComputerName)\adminshared$"
                        Identity      = $User.Username
                    }
                
                    # Clone User permissions from the administrator account.
                    Set-ADUser @Splathash
                    Get-ADuser -identity "Administrator" -properties memberof | Select-Object memberof -expandproperty memberof | Add-AdGroupMember -Members $User.Username
                    Write-Host "User $($User.username) Altered in AD" -ForegroundColor Yellow
                }
                else {
                
                    # Build Splat Hash.
                    $Splathash = @{
                        Name                  = "$($User.fname) $($User.lname)"
                        Displayname           = "$($User.fname) $($User.lname)"
                        Path                  = "OU=AP Admin Accounts,$($DomainDName)"
                        Surname               = $User.lname
                        GivenName             = $User.fname
                        Samaccountname        = "$($User.username)"
                        UserPrincipalName     = "$($user.username)@$($ClientEmailDomain)"
                        EmailAddress          = "$($user.username)@$($ClientEmailDomain)"
                        Homedrive             = "H:"
                        AccountPassword       = ConvertTo-SecureString -String "!!AP:2018!!" -AsPlainText -Force
                        Enabled               = $True
                        PasswordNeverExpires  = $True
                        ChangePasswordAtLogon = $False
                        ProfilePath           = "\\$($ComputerName)\adminprofiles$\$($User.username)"
                        HomeDirectory         = "\\$($ComputerName)\adminshared$"
                    }

                    # Add User
                    New-ADUser @Splathash
                    Get-ADuser -identity "Administrator" -properties memberof | Select-Object memberof -expandproperty memberof | Add-AdGroupMember -Members $User.Username
                    Write-Host "User $($User.username) Created in AD" -ForegroundColor Green
                }

                # Add in users area on desktop.
                New-Item -ItemType Directory -Force -Path "$($AdminSharedFolder)\Desktop\My Work\$($User.username)"
            }

            # Add line break, tell the users wnhats going on
            Write-Host ""
            Write-Host "Removing old users" -ForegroundColor Green
            Write-Host ""

            # Import list into object for further processing
            [object]$APRemoveUsers = Import-Csv -Path $APRemoveList -Delimiter ","

            # Iterate through the list and remove any that are found
            foreach ($User in $APRemoveUsers) {

                # Check to see if user exists on system
                if (Test-UserExists -Username "$($User.Username)" -SearchMode "Local") {
                    Remove-ADUser -Identity $User.Username -Confirm:$false
                    Write-Host "User $($User.username) expunged from this system" -ForegroundColor Red
                }
                else {
                    Write-Host "User $($User.username) already expunged from this system" -ForegroundColor Yellow
                }
            }
        }
    }

    END {

        # Cleanup if required.
        If ($ExitMode -eq "Cleanup") {

            # Tell the users what is going on.
            Write-Host ""
            Write-Host ""
            Write-Host "Deleting files / scratch folders used by script" -ForegroundColor Magenta

            # Remove Roadie output folder.
            Remove-Item "C:\$RoadieGUID" -Recurse -Force
        }
    }
}

function Update-GPOTemplates {

    # Get parameters
    Param(
        [parameter(Mandatory = $true, HelpMessage = "Please provide the name of the script calling this function", position = 0)][string]$ScriptName,
        [ValidateSet("Interactive", "Service")][parameter(Mandatory = $true, HelpMessage = "Please choose output mode", position = 1)][string]$OutputMode,
        [ValidateSet("Cleanup", "LeaveBehind")][parameter(Mandatory = $true, HelpMessage = "Please choose Exit mode", position = 2)][string]$ExitMode
    )

    BEGIN {

        # Setup variables.
        [bool]$OnDomainController = $false

        # Check we are on a permitted machine for this function. (1st DC in domain)
        if (Test-MachineName "Filter" "DomainController" $ComputerName) {
            
            # Mark that we are on correct machine.
            $OnDomainController = $true
        }

        # Setup additional variables.
        [string]$RoadieGUID = "fdbe5abe-7f38-4fd9-88bd-008d28baafb6"
        [string]$RoadieResourceBundleUrl = "https://deploy.theaccesspoint.co.uk/$($RoadieGUID)/resources.zip"
        [string]$RoadieTemp = "C:\$($RoadieGUID)\roadie.zip"
        [string]$RoadieResourceBundleOutput = "C:\$($RoadieGUID)"
        [string]$GPOManifestFile = "C:\$RoadieGUID\GPO_Manifest.txt"
        [bool]$GPOManifestPresent = $false
        [bool]$GPOManifestFolderPresent = $false

        # Process this block if we are on a domain controller.
        if ($OnDomainController -eq $true) {

            # Check for required files in guid folder (Roadie) - flag status
            $GPOManifestPresent = Test-Path -Path "$($GPOManifestFile)" -PathType Leaf

            # Download and extract if file not present.
            if ($GPOManifestPresent -eq $false) {

                # Setup scratch folders.
                Write-Host "Setting up Scratch Folders" -ForegroundColor Green
                New-Item -ItemType Directory -Force -Path "C:\$($RoadieGUID)"

                # log / display Event.
                Show-Output -Message "Manifest File is NOT present in expected location, downloading resource file." -OutputFormat "Warning" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

                # Build Resource Splat hash.
                $ResourceSplathash = @{
                    ScriptName       = "$($ScriptName)"
                    OutputMode       = $OutputMode
                    DownloadURL      = "$($RoadieResourceBundleUrl)"
                    DownloadLocation = "$($RoadieTemp)"
                    ExtractionPath   = "$($RoadieResourceBundleOutput)"
                    ResourcePackName = "Roadie"
                }

                # Download and extract resource pack
                Initialize-ResourcePack @ResourceSplathash
            }
            else {

                # log / display Event.
                Show-Output -Message "Manifest File is present in expected location, no need to download resource file." -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
            }
        }
    }

    PROCESS {

        # Process this block if we are on a domain controller.
        if ($OnDomainController -eq $true) {

            # Add line break, tell the users wnhats going on
            Write-Host ""
            Write-Host "Copying built in policy definitions to central store." -ForegroundColor Green

            # Copy items from default store to central policy store.
            Copy-Item "C:\Windows\PolicyDefinitions\" "C:\Windows\SYSVOL\domain\Policies\" -Recurse -Force

            # Import list into object for further processing
            [object]$GPOManifest = Import-Csv -Path $GPOManifestFile -Delimiter ","

            # Iterate through the list and copy the GPOs listed into the central policy definition store.
            foreach ($ManifestLine in $GPOManifest) {

                # Check folder exists.
                $GPOManifestFolderPresent = Test-Path "C:\$($RoadieGUID)\$($ManifestLine)"

                # Copy folder to central policy definitions store (If it exists)
                if ($GPOManifestFolderPresent -eq $true) {

                    # log / display Event.
                    Write-Host ""
                    Show-Output -Message "GPO Folder is present in expected location, copying files." -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
                    
                    # Copy in additional policies in (Google / Acrobat / Office 2016 / New windows GPO Templates)
                    Copy-Item "C:\$($RoadieGUID)\$($ManifestLine)\*" "C:\Windows\SYSVOL\domain\Policies\PolicyDefinitions" -Recurse -Force
                }
                else {

                    # log / display Event.
                    Write-Host ""
                    Show-Output -Message "GPO Folder is NOT present in expected location" -OutputFormat "Warning" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
                }

                # Reset folder presence check
                $GPOManifestFolderPresent = $false
            }
        }
    }

    END {

        # Cleanup if required.
        If ($ExitMode -eq "Cleanup") {

            # Tell the users what is going on.
            Write-Host ""
            Write-Host ""
            Write-Host "Deleting files / scratch folders used by script" -ForegroundColor Magenta

            # Remove Roadie output folder.
            Remove-Item "C:\$RoadieGUID" -Recurse -Force
        }
    }
}