# External leaks
function Send-FTP {

    param (
        [Parameter(Mandatory = $true, HelpMessage = "Source file path", Position = 0)][string]$SourceFilePath,
        [Parameter(Mandatory = $true, HelpMessage = "Destination file path", Position = 1)][string]$DestFilePath,
        [Parameter(Mandatory = $true, HelpMessage = "FTP User Username", Position = 2)][string]$FTPUser,
        [Parameter(Mandatory = $true, HelpMessage = "FTP user Password", Position = 3)][string]$FTPPass,
        [Parameter(Mandatory = $true, HelpMessage = "Address of FTP host", Position = 4)][string]$FTPHost,
        [parameter(Mandatory = $true, HelpMessage = "Please provide the name of the script calling this function", position = 5)][string]$ScriptName,
        [ValidateSet("Interactive", "Service")][Parameter(Mandatory = $true, HelpMessage = "Please choose output mode.", Position = 6)][string]$OutputMode
    )

    BEGIN {

        # Setup variables.
        [string]$OldErrorPreference = $ErrorActionPreference
        [bool]$FileExists = $true
        [bool]$ServerExists = $true
        [bool]$ProcessUpload = $true

        # Override error preference
        $ErrorActionPreference = 'stop'

        # Check filepath to ensure file is there.
        $FileExists = Test-Path -Path "$($SourceFilePath)" -PathType Leaf

        # Test server exists
        $ServerExists = Test-NetConnection -ComputerName $FTPHost -Port 21 -InformationLevel Quiet

        # Initiate Webclient
        $webclient = New-Object -TypeName System.Net.WebClient

        # Build FTP Upload string
        [string]$FTPCommand = "ftp://$($FTPUser):$($FTPPass)@$($FTPHost)/$($DestFilePath)"

        # Build URI Object
        $uri = New-Object -TypeName System.Uri -ArgumentList $FTPCommand

        # Check if script should proceed.
        if (($FileExists -eq $false) -or ($ServerExists -eq $false)) {

            # Mark as do not proceed if we have a false on either value.
            $ProcessUpload = $false
        }
    }

    PROCESS {

        # See if we are to attempt the upload.
        if ($ProcessUpload -eq $true) {

            # log / display Event.
            Show-Output -Message "Attmepting to Upload: $($SourceFilePath) to Server: $($FTPHost)" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

            # Let the user know what is going on.
            Write-Host "Uploading: " -NoNewline; Write-Host "$SourceFilePath" -ForegroundColor Yellow -NoNewline; Write-Host " to Host: " -NoNewline; Write-Host $FTPHost -ForegroundColor Yellow -NoNewline
            Write-Host ""
            Write-Host "Source File Path      : " -NoNewline; Write-Host "$($SourceFilePath)" -ForegroundColor Magenta
            Write-Host "Destination File Path : " -NoNewline; Write-Host "$($DestFilePath)" -ForegroundColor Magenta
            Write-Host "FTP Host              : " -NoNewline; Write-Host "$($FTPHost)" -ForegroundColor Magenta

            Try {

                # Upload File.
                $webclient.UploadFile($uri, $SourceFilePath)

                # log / display Event.
                Show-Output -Message "Upload to Server: $($FTPHost) Complete" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
            }
            
            Catch {

                # Catch exception and show error.
                Show-Output -Message "Upload to Server: $($FTPHost) was NOT Completed" -OutputFormat "Error" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
                Show-Output -Message "$($_.Exception.Message)" -OutputFormat "Error" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
            }
            Finally {
                # Set error preference back to what it was.
                $ErrorActionPreference = $OldErrorPreference
            }
        }
        else {

            # Report on source file error.
            if ($FileExists -eq $false) {

                # log / display Event.
                Show-Output -Message "Source file: $($SourceFilePath) was NOT found." -OutputFormat "Error" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
            }

            # Report on server error.
            if ($ServerExists -eq $false) {

                # log / display Event.
                Show-Output -Message "Server: $($FTPHost) was NOT found / online" -OutputFormat "Error" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
            }
        }
    }

    END {

        # Dispose of object.
        $webclient.Dispose()
    }
}

function Get-WebFile {

    param (
        [Parameter(Mandatory = $true, HelpMessage = "URL to download file from (HTTP / HTTPS)", Position = 0)][string]$UrlPath,
        [Parameter(Mandatory = $true, HelpMessage = "Location on disk to save the file to", Position = 1)][string]$DownloadPath,
        [parameter(Mandatory = $true, HelpMessage = "Please provide the name of the script calling this function", position = 2)][string]$ScriptName,
        [ValidateSet("Interactive", "Service")][Parameter(Mandatory = $true, HelpMessage = "Please choose output mode.", Position = 3)][string]$OutputMode
    )
    
    BEGIN {

        # Setup variables.
        [string]$OldErrorPreference = $ErrorActionPreference
        [bool]$ErrorOccured = $false

        # Override error preference.
        $ErrorActionPreference = 'stop'

        # Setup web objects, catch and handle errors as there is much to go wrong here
        Try {
            $systemuri = New-Object "System.Uri" "$UrlPath"
            $webrequest = [System.Net.HttpWebRequest]::Create($systemuri)
            $webrequest.set_Timeout(15000) #15 second timeout
            $webresponse = $webrequest.GetResponse()
            $totalLength = [System.Math]::Floor($webresponse.get_ContentLength() / 1024)
            $responseStream = $webresponse.GetResponseStream()
        }
        Catch {

            # Tell the users what is going on.
            if ($_.Exception | Select-String -Pattern "error: (404)" -SimpleMatch) {
                Write-Host ""
                Write-Host "Address incorrect, unable to download file." -ForegroundColor Red
            }
            else {
                Write-Host ""
                Write-Host "Server Error" -ForegroundColor Red
            }

            # Mark that an error has occured
            $ErrorOccured = $true
        }

        if ($ErrorOccured -eq $false) {
            
            Try {
                $targetStream = New-Object -TypeName System.IO.FileStream -ArgumentList $DownloadPath, Create
            }
            Catch {
                
                # Tell the users what is going on.
                Write-Host ""
                Write-Host "Unable to save file." -ForegroundColor Red -NoNewline; Write-Host " Check the file path and that you have appropriate permissions to write to that location."

                # Mark that an error has occured
                $ErrorOccured = $true
            }

            $buffer = new-object byte[] 512KB
            $count = $responseStream.Read($buffer, 0, $buffer.length)
            $downloadedBytes = $count

            # Start stopwatch
            $stopwatch = [system.diagnostics.stopwatch]::StartNew()
        }
    }

    PROCESS {

        #Check that all is good upto this point.
        if ($ErrorOccured -eq $false) {

            # New line.
            Write-Host ""

            # Attempt the download
            Try {
                while ($count -gt 0) {

                    # Process file.
                    $targetStream.Write($buffer, 0, $count)
                    $count = $responseStream.Read($buffer, 0, $buffer.length)
                    $downloadedBytes = $downloadedBytes + $count
                    $PercentProgress = $('{0:N2}' -f [math]::Round(((([System.Math]::Floor($downloadedBytes / 1024)) / $totalLength) * 100), 2))

                    # Write to console what is going on
                    [System.Console]::CursorLeft = 0
                    [System.Console]::Write("Downloading File: '$($UrlPath.split('/') | Select-Object -Last 1)' Downloaded ($([System.Math]::Floor($downloadedBytes / 1024))K of $($totalLength)K): $($PercentProgress)% Complete")
                }
            }
            Catch {

                # Tell the users what is going on.
                Write-Host "An error occured while transferring the file from the remote host." -ForegroundColor Red
            }
            Finally {
                # Set error preference back to what it was.
                $ErrorActionPreference = $OldErrorPreference
            }
        }
    }

    END {

        # Stop stopwatch.
        $stopwatch.stop()

        # Display results depending on outcome
        if ($ErrorOccured -eq $false) {

            # Tell the user what is going on.
            Write-Host ""
            Write-Host ""

            if ($stopwatch.elapsed.minutes -eq 0) {
                Write-Host "File: '" -NoNewline; Write-Host "$($UrlPath.split('/') | Select-Object -Last 1)" -ForegroundColor Magenta -NoNewline; Write-Host "' Downloaded in $($stopwatch.elapsed.seconds) Second(s)"
            }
            else {
                Write-Host "File: '" -NoNewline; Write-Host "$($UrlPath.split('/') | Select-Object -Last 1)" -ForegroundColor Magenta -NoNewline; Write-Host "' Downloaded in $($stopwatch.elapsed.minutes) Minute(s) and $($stopwatch.elapsed.seconds) Second(s)"
            }

            Write-Host "File Size: $('{0:N2}' -f [math]::Round(($totalLength / 1024))) MB"
            Write-Host "Transfer Rate: $('{0:N2}' -f [math]::Round($($totalLength/1024)/($stopwatch.elapsed.totalseconds),2)) MB/Sec"
        }
        else {
            
            # Tell the user what is going on.
            Write-Host ""
            Write-Host "An error occured that resulted in the download not being completed, please see above this message for further diagnostics" -ForegroundColor Red
        }

        # remove objects used in function.
        try {
            $targetStream.Flush()
            $targetStream.Close()
            $targetStream.Dispose()
            $responseStream.Dispose()
        }
        catch { }
    }
}

function Send-Email {

    param (
        [Parameter(Mandatory = $true, HelpMessage = "Address email is to appear from.", Position = 0)][string]$FromAddress,
        [Parameter(Mandatory = $true, HelpMessage = "Address(es) email is to be sent to.", Position = 1)][object]$ToAddresses,
        [Parameter(Mandatory = $false, HelpMessage = "Address(es) email is to be CC to.", Position = 2)][object]$CCAddresses,
        [Parameter(Mandatory = $false, HelpMessage = "Address(es) email is to be BCC to.", Position = 3)][object]$BCCAddresses,
        [Parameter(Mandatory = $true, HelpMessage = "Subject of Email.", Position = 4)][string]$EmailSubject,
        [Parameter(Mandatory = $true, HelpMessage = "Body of Email.", Position = 5)][string]$EmailBody,
        [Parameter(Mandatory = $true, HelpMessage = "Is Email HTML.", Position = 6)][bool]$HTMLEmail,
        [Parameter(Mandatory = $false, HelpMessage = "Email Attachment(s).", Position = 7)][object]$Attachments,
        [ValidateSet("Normal", "High", "Low")][Parameter(Mandatory = $true, HelpMessage = "Email Priority.", Position = 8)][string]$EmailPriority,
        [ValidateSet("None", "OnSuccess", "OnFailure", "Delay", "Never")][Parameter(Mandatory = $true, HelpMessage = "Delivery Notifications.", Position = 9)][string]$DeliveryNotifications,
        [Parameter(Mandatory = $true, HelpMessage = "Address of SMTP Server to use.", Position = 10)][string]$SMTPServer,
        [parameter(Mandatory = $true, HelpMessage = "Please provide the name of the script calling this function", position = 11)][string]$ScriptName,
        [ValidateSet("Interactive", "Service")][Parameter(Mandatory = $true, HelpMessage = "Please choose output mode.", Position = 12)][string]$OutputMode
    )

    BEGIN {

        # Setup Variables.
        [string]$EMailValidationPattern = "^((?:\w+)|(?:\w+)\.(?:\w+))@([a-z0-9\-]{2,}|[a-z0-9\-]{2,}\.[a-z0-9]{2,})(\.[a-z]{2,}){1,2}$"
        [bool]$FromAddressValid = $true
        [bool]$ToAddressValid = $true
        [bool]$CCAddressValid = $true
        [bool]$BCCAddressValid = $true
        [bool]$AttachmentValid = $false
        [bool]$SMTPServerValid = $false
        [bool]$ProcessMessage = $true

        # Check From address.
        if ($FromAddress | Select-String -Pattern "$($EMailValidationPattern)") {
            $FromAddressValid = $true

            # log / display Event.
            Show-Output -Message "From address ($($FromAddress)) valid." -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
        } else {
            $FromAddressValid = $false

            # log / display Event.
            Show-Output -Message "From address ($($FromAddress)) NOT valid." -OutputFormat "Error" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
        }

        # Check To Address(es) (Iterate through object).
        foreach ($ToAddress in $ToAddresses) {
            
            # Check each To address (Only look for fails).
            if ($ToAddress | Select-String -Pattern "$($EMailValidationPattern)") {
                $ToAddressValid = $true

                # log / display Event.
                Show-Output -Message "To address ($($ToAddress)) valid." -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
            }
            else {
                $ToAddressValid = $false
                $ProcessMessage = $false

                # log / display Event.
                Show-Output -Message "To address ($($ToAddress)) NOT valid." -OutputFormat "Error" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
            }
        }

        # Check to see if there are any CC addresses to be checked.
        if ($CCAddresses.count -ge 1) {

            # Check CC Address(es) (Iterate through object).
            foreach ($CCAddress in $CCAddresses) {
                
                # Check each To address (Only look for fails).
                if ($CCAddress | Select-String -Pattern "$($EMailValidationPattern)") {
                    $CCAddressValid = $true

                    # log / display Event.
                    Show-Output -Message "CC address ($($CCAddress)) valid." -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
                }
                else {
                    $CCAddressValid = $false
                    $ProcessMessage = $false

                    # log / display Event.
                    Show-Output -Message "CC address ($($CCAddress)) NOT valid." -OutputFormat "Error" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
                }
            }
        }
        else {

            # log / display Event.
            Show-Output -Message "No CC address(es) specified." -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
        }
        
        # Check to see if there are any BCC addresses to be checked.
        if ($BCCAddresses.count -ge 1) {

            # Check BCC Address(es) (Iterate through object).
            foreach ($BCCAddress in $BCCAddresses) {
                
                # Check each To address (Only look for fails).
                if ($BCCAddress | Select-String -Pattern "$($EMailValidationPattern)") {
                    $BCCAddressValid = $true

                    # log / display Event.
                    Show-Output -Message "BCC address ($($BCCAddress)) valid." -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
                }
                else {
                    $BCCAddressValid = $false
                    $ProcessMessage = $false

                    # log / display Event.
                    Show-Output -Message "BCC address ($($BCCAddress)) NOT valid." -OutputFormat "Error" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
                }
            }
        }
        else {

            # log / display Event.
            Show-Output -Message "No BCC address(es) specified." -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
        }

        # Check attachment paths (if attachments have been specified).
        if ($Attachments.count -ge 1) {

            # Test file paths.
            foreach ($Attachment in $Attachments) {
                if ((Test-Path -Path "$($Attachment)" -PathType Leaf) -eq $true) {
                    $AttachmentValid = $true
                }
                else {
                    $AttachmentValid = $true
                    $ProcessMessage = $false
                }
            }
        }
        else {

            # log / display Event.
            Show-Output -Message "No attachment(s) specified." -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
        }

        # Verify that the SMTP server is online.
        $SMTPServerValid = Test-NetConnection -ComputerName "$($SMTPServer)" -Port 25 -InformationLevel Quiet

        if ($SMTPServerValid -eq $false) {
            $ProcessMessage = $false

            # log / display Event.
            Show-Output -Message "SMTP Server specified is NOT valid." -OutputFormat "Error" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
        }
        else {

            # log / display Event.
            Show-Output -Message "SMTP Server specified is valid." -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
        }
    }

    PROCESS {

        # Check that we are good to proceed.
        if ($ProcessMessage -eq $true) {

            # Build email message splat.
            $EmailSplathash = @{
                From                       = "$($FromAddress)"
                To                         = $ToAddresses
                CC                         = $CCAddresses
                BCC                        = $BCCAddresses
                Subject                    = "$($EmailSubject)"
                Body                       = "$($EmailBody)"
                BodyAsHtml                 = $HTMLEmail
                Attachments                = $Attachments
                Priority                   = "$($EmailPriority)"
                DeliveryNotificationOption = "$($DeliveryNotifications)"
                SmtpServer                 = "$($SMTPServer)"
            }

            # Splat modificataions to remove zero length items.
            if ($CCAddresses.count -lt 1) {
                $EmailSplathash.Remove("CC")
            }

            if ($BCCAddresses.count -lt 1) {
                $EmailSplathash.Remove("BCC")
            }

            if ($Attachments.count -lt 1) {
                $EmailSplathash.Remove("Attachments")
            }

            # Send email message.
            Send-MailMessage @EmailSplathash

            # log / display Event.
            Show-Output -Message "Email Sent." -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
        }
        else {

            # log / display Event.
            Show-Output -Message "Email was NOT Sent, Please review previous messages / log events." -OutputFormat "Error" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
        }
    }

    END {}
}

function Send-TeamsMessage {

    param (
        [Parameter(Mandatory = $true, HelpMessage = "URL for teams channel webhook.", Position = 0)][string]$WebHookPath,
        [Parameter(Mandatory = $true, HelpMessage = "Title of message to post", Position = 1)][string]$MessageTitle,
        [Parameter(Mandatory = $true, HelpMessage = "Body of message to post", Position = 2)][string]$MessageBody,
        [parameter(Mandatory = $true, HelpMessage = "Please provide the name of the script calling this function", position = 3)][string]$ScriptName,
        [ValidateSet("Interactive", "Service")][Parameter(Mandatory = $true, HelpMessage = "Please choose output mode.", Position = 4)][string]$OutputMode
    )

    BEGIN {

        # Setup objects / variables.
        [bool]$HostStatus = $true
        [string]$HostName = $WebHookPath -replace "^(https:\/\/)((?:\w+\.){1,}\w+)(.+)$", '$2'

        # Build message JSON payload
        [string]$JSONPayload = @{title="$($MessageTitle)";textFormat="markdown";text="$($MessageBody)"} | ConvertTo-Json -Compress

        # Check for endpoint validity
        if (!(Test-NetConnection -ComputerName "$($HostName)" -Port 443)) {

            # Mark host as unreachable on port 443
            $HostStatus = $false
        }
    }

    PROCESS {

        # $MessageBody = "Test 6  
        # ![Typing Cat](https://media.giphy.com/media/juSCTzqDAV1Xq/giphy.gif)  
        # shitposting commences"

        # Process if host status is true.
        if ($HostStatus -eq $true) {
            
            # Send teams message (JSON formatted with markdown capability), if endpoint is available
            Invoke-RestMethod -Method post -ContentType 'Application/Json' -Body "$($JSONPayload)" -Uri $WebHookPath
        }
    }

    END {}
}

function Get-ZabbixData {
    
    param (
        [Parameter(Mandatory = $true, HelpMessage = "Security Token (Obtained in seperate function).", Position = 0)][string]$SecurityToken,
        [ValidateSet("host_get.exact", "host_get.groupid", "host_get.search", "hostgroup_get.exact", "hostgroup_get.groupid", "hostgroup_get.search", "item_get.search")][Parameter(Mandatory = $true, HelpMessage = "Request Method", Position = 1)][string]$RequestMethod,
        [Parameter(Mandatory = $false, HelpMessage = "Host ID (Used with item_get.host)", Position = 2)][int]$HostID,
        [Parameter(Mandatory = $true, HelpMessage = "Value to search for.", Position = 3)][string]$SearchValue,
        [parameter(Mandatory = $true, HelpMessage = "Please provide the name of the script calling this function", position = 4)][string]$ScriptName,
        [ValidateSet("Interactive", "Service")][Parameter(Mandatory = $true, HelpMessage = "Please choose output mode.", Position = 5)][string]$OutputMode
    )

    BEGIN {

        # Setup variables.
        [object]$TranslateMatrix = @()
        [bool]$HostStatus = $true
        [bool]$DataProcessed = $false
        [string]$BaseURL = "zabbix.ap.systems"
        [object]$APISearchMode = ""
        [string]$APISearchTarget = ""

        # Build translation matrix.
        $TranslateMatrix += [PSCustomObject]@{Regular = "exact"; Zabbix = "filter" }
        $TranslateMatrix += [PSCustomObject]@{Regular = "groupid"; Zabbix = "groupids" }
        $TranslateMatrix += [PSCustomObject]@{Regular = "hostid"; Zabbix = "hostids" }
        $TranslateMatrix += [PSCustomObject]@{Regular = "search"; Zabbix = "search" }

        # Get Zabbix server availability.
        $HostStatus = Test-NetConnection -ComputerName "$($BaseURL)" -Port 443 -InformationLevel Quiet
    }

    PROCESS {

        # Process item based on outcome of host check.
        if ($HostStatus -eq $true) {

            # log / display Event
            Show-Output -Message "Attempting to connect to Zabbix API (Host: https://$($BaseURL)), using method: $($RequestMethod)." -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

            # Setup search parameters.
            $APISearchTarget = ($RequestMethod) -replace "^(\w+)\.(\w+)$", '$2'
            $APISearchMode = $TranslateMatrix | Where-Object { $_.Regular -eq "$($APISearchTarget)" } | Select-Object "Zabbix"

            # log / display Event
            Show-Output -Message "Search Mode: $($SearchMode) translated as $($APISearchMode.Zabbix). Search Target: $($APISearchTarget). Search Value: $($SearchValue)" -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

            if ($RequestMethod -like "host_get.*") {

                # Build JSON query.
                $params = @{
                    body    = @{
                        "jsonrpc" = "2.0"
                        "method"  = "host.get"
                        "params"  = @{
                            "output" = "extend"
                            "$($APISearchMode.Zabbix)" = @{ "host" = "$($SearchValue)" }
                        }
                        "auth"    = "$($SecurityToken)"
                        "id"      = 1
                    } | ConvertTo-Json
                    uri     = "https://$BaseURL/api_jsonrpc.php"
                    headers = @{"Content-Type" = "application/json" }
                    method  = "Post"
                }
            }
            elseif ($RequestMethod -like "hostgroup_get.*") {

                # Switch search targets
                if ($APISearchTarget -ne "groupid") {
                    $APISearchTarget = "name"
                }

                # Build JSON query.
                $params = @{
                    body    = @{
                        "jsonrpc" = "2.0"
                        "method"  = "hostgroup.get"
                        "params"  = @{
                            "output" = "extend"
                            "$($APISearchMode.Zabbix)" = @{ "$APISearchTarget" = "$($SearchValue)" }
                        }
                        "auth"    = "$($SecurityToken)"
                        "id"      = 1
                    } | ConvertTo-Json
                    uri     = "https://$BaseURL/api_jsonrpc.php"
                    headers = @{"Content-Type" = "application/json" }
                    method  = "Post"
                }
            }
            elseif ($RequestMethod -like "item_get.*") {

                # Build JSON query.
                $params = @{
                    body    = @{
                        "jsonrpc" = "2.0"
                        "method"  = "item.get"
                        "params"  = @{
                            "output"    = "extend"
                            "hostids"   = "$($HostID)"
                            "$($APISearchMode.Zabbix)" = @{ "key_" = "$($SearchValue)" }
                            "sortfield" = "name"
                        }
                        "auth"    = "$($SecurityToken)"
                        "id"      = 1
                    } | ConvertTo-Json
                    uri     = "https://$BaseURL/api_jsonrpc.php"
                    headers = @{"Content-Type" = "application/json" }
                    method  = "Post"
                }
            }

            # Process JSON query and store result for further processing.
            $APIRequestResult = Invoke-WebRequest @params

            # Set data processed as true if status is 200
            if ($APIRequestResult.StatusCode -eq "200") {

                $DataProcessed = $true

                # log / display Event
                Show-Output -Message "Connection to Zabbix API (Host: https://$($BaseURL)), using method: $($RequestMethod) was successful, data may not have been returned, please check further messages." -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
            }
            else {

                # log / display Event
                Show-Output -Message "Connection to Zabbix API (Host: https://$($BaseURL)), using method: $($RequestMethod) was NOT successful. Status code: $($APIRequestResult.StatusCode)" -OutputFormat "Error" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
            }
        }
        else {

            # log / display Event.
            Show-Output -Message "Host https://$($BaseURL) was not available, no processing was carried out." -OutputFormat "Error" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 920
        }
    }

    END {

        # Check to see if data was retreived.
        if ($DataProcessed -eq $true) {
        
            # Extract data from query results.
            $APIData = $APIRequestResult.Content | ConvertFrom-Json

            if (($APIData.error | Measure-Object).Count -gt 0) {

                # log / display Event
                Show-Output -Message "An Error Occured. Code: $($APIData.error.code), Message: $($APIData.error.message), Data: $($APIData.error.data)." -OutputFormat "Error" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0

                # Return data.
                return $APIData.error
            }

            if (($APIData.result | Measure-Object).Count -gt 0) {

                # log / display Event
                Show-Output -Message "Data was retreived from the API." -OutputFormat "Information" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
             
                # Return data.
                return $APIData.result
            }
            elseif (($APIData.result | Measure-Object).Count -eq 0) {
                
                # log / display Event
                Show-Output -Message "Data was retreived from the API but may be blank." -OutputFormat "Warning" -OutputMode $OutputMode -EventSource "$($ScriptName)" -EventIDOverride 0
             
                # Return data.
                return $APIData.result
            }
        }
        else {

            # Return data.
            return "No Data Received"
        }
    }
}